<?php

namespace App\Http\Controllers;

use App\Models\Course;
use App\Models\University;
use App\Models\CourseDescription;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use App\Models\CourseCategory;


class CourseController extends Controller
{
    public function index()
    {
        $courses = Course::with('university')->get();
        return view('backend.courses.index', compact('courses'));
    }

public function create()
{
    $universities = University::all();
    $categories = CourseCategory::all(); // Fetch all course categories
    return view('backend.courses.create', compact('universities', 'categories'));
}

public function store(Request $request)
{


    $request->validate([
        'university_id' => 'required|exists:universities,id',
        'category_id' => 'required|exists:course_categories,id', // Validate category selection
        'title' => 'required|string',
        'deliverable' => 'nullable|string',
        'duration' => 'nullable|string',
        'eligibility_criteria' => 'nullable|string',
        'subjects' => 'nullable|string',
        'image' => 'nullable|image|max:2048',
    ]);

    $course = new Course($request->all());

    // Generate unique cslug
    $cslug = Str::slug($request->title);
    if (Course::where('cslug', $cslug)->exists()) {
        $cslug .= '-' . uniqid();
    }
    $course->cslug = $cslug;

        if ($request->hasFile('image')) {
            $file = $request->file('image');
            $filename = uniqid() . '.' . $file->getClientOriginalExtension();
            $course->image = $file->storeAs('courses', $filename, 'public');
        }

    $course->save();

    // Handle course descriptions if provided
    if ($request->has('descriptions')) {
        foreach ($request->descriptions['title'] as $index => $title) {
            $description = new CourseDescription([
                'title' => $title,
                'description' => $request->descriptions['description'][$index],
                'key_points' => $request->descriptions['key_points'][$index],
                'course_id' => $course->id,
            ]);
            $description->save();
        }
    }

    return redirect()->route('backend.courses.index')
                     ->with('success', 'Course created successfully.');
}

public function update(Request $request, Course $course)
{


    $request->validate([
        'university_id' => 'required|exists:universities,id',
        'category_id' => 'required|exists:course_categories,id', // Validate category
        'title' => 'required|string',
        'cslug' => 'nullable|unique:courses,cslug,' . $course->id,
        'popular' => 'nullable|boolean',
    ]);

    // --- Update course main fields ---
    $data = $request->only([
        'university_id',
        'category_id', // Include category in the update
        'title',
        'deliverable',
        'duration',
        'eligibility_criteria',
        'subjects'
    ]);

    $data['cslug'] = $request->cslug ?? Str::slug($request->title);
    $data['popular'] = $request->has('popular') ? 1 : 0;

    // --- Handle image update ---
   if ($request->hasFile('image')) {
    if ($course->image) {
        Storage::disk('public')->delete($course->image);
    }
    $file = $request->file('image');
    $filename = uniqid() . '.' . $file->getClientOriginalExtension();
    $data['image'] = $file->storeAs('courses', $filename, 'public');
}

    $course->update($data);

    // --- Handle course descriptions update ---
    if ($request->has('descriptions')) {
        // Remove old descriptions first
        $course->descriptions()->delete();

        // Insert updated descriptions
        foreach ($request->descriptions['title'] as $index => $title) {
            $description = new CourseDescription([
                'title' => $title,
                'description' => $request->descriptions['description'][$index],
                'key_points' => $request->descriptions['key_points'][$index],
                'course_id' => $course->id,
            ]);
            $description->save();
        }
    }

    return redirect()->route('backend.courses.index')
                     ->with('success', 'Course updated successfully with descriptions.');
}




public function edit(Course $course)
{
    $universities = University::all();
    $categories = CourseCategory::all(); // Fetch all course categories
    return view('backend.courses.edit', compact('course', 'universities', 'categories'));
}

    public function destroy(Course $course)
    {
        if ($course->image) {
            Storage::disk('public')->delete($course->image);
        }
        $course->delete();

        return back()->with('success', 'Course deleted successfully.');
    }

    public function showUniversityCourses($slug)
    {
        $university = University::where('slug', $slug)->firstOrFail();
        
        $courses = $university->courses;
        $universities = University::all();

        return view('frontend.university_courses', compact('university', 'courses', 'universities'));
    }

    public function show($cslug)
    {
        $course = Course::where('cslug', $cslug)->firstOrFail();
        return view('frontend.course_details', compact('course'));
    }

    public function discoverCourse($cslug)
    {
        $course = Course::where('cslug', $cslug)->firstOrFail();
        $universities = University::all();

        return view('frontend.course_details', compact('course', 'universities'));
    }

    public function updatePopular(Request $request, Course $course)
    {
        $course->popular = $request->has('popular') ? 1 : 0;
        $course->save();

        return back()->with('success', 'Course popularity status updated.');
    }

    public function updateDescriptions(Request $request, $courseId)
    {
        $course = Course::findOrFail($courseId);

        $request->validate([
            'title' => 'required|array',
            'description' => 'required|array',
            'key_points' => 'required|array',
        ]);

        $course->descriptions()->delete();

        foreach ($request->title as $index => $title) {
            $description = new CourseDescription([
                'title' => $request->title[$index],
                'description' => $request->description[$index],
                'key_points' => $request->key_points[$index],
                'course_id' => $course->id,
            ]);
            $course->descriptions()->save($description);
        }

        return redirect()->route('backend.courses.index')
                         ->with('success', 'Course descriptions updated successfully.');
    }

    public function exploreAll(Request $request)
{
    // Retrieve all universities and categories for filtering
    $universities = University::all();
    $categories = CourseCategory::all();

    // Retrieve all courses, applying filters if present
    $courses = Course::with(['university', 'category'])
                     ->when($request->category_id, function ($query) use ($request) {
                         return $query->where('category_id', $request->category_id);
                     })
                     ->when($request->university_id, function ($query) use ($request) {
                         return $query->where('university_id', $request->university_id);
                     })
                     ->get();

    return view('frontend.courses_explore', compact('courses', 'universities', 'categories'));
}

}
