<?php

namespace App\Http\Controllers;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\Request;
use App\User;
use App\Models\Order;
use App\Models\Product;
use App\Models\ProductReview;
use App\Models\PostComment;
use App\Rules\MatchOldPassword;
use Hash;

class HomeController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Show the application dashboard.
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */
    public function index(){
        return view('user.index');
    }

    public function profile(){
        $profile=Auth()->user();
        return view('user.users.profile')->with('profile',$profile);
    }

    public function profileUpdate(Request $request, $id)
{
    $user = User::findOrFail($id);

    // Validate the request
    $request->validate([
        'name' => 'required|string|max:255',
        'address1' => 'required|string|max:255',
        'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
    ]);

    $data = $request->all();

    // Check if a photo has been uploaded
    if ($request->hasFile('photo')) {
        // Delete the previous image
        $oldImagePath = public_path('backend/img/' . $user->photo);
        if (File::exists($oldImagePath)) {
            File::delete($oldImagePath);
        }

        // Upload the new image
        $image = $request->file('photo');
        $imageName = time() . '.' . $image->getClientOriginalExtension();
        $image->move(public_path('backend/img/'), $imageName);

        // Update user photo
        $user->photo = $imageName;
    } else {
        // If no new photo is uploaded, keep the existing photo
        $data['photo'] = $user->photo;
    }

    // Update other user data
    $user->name = $data['name'];
    $user->address1 = $data['address1'];

    // Save the updated user data
    $status = $user->save();

    if ($status) {
        $request->session()->flash('success', 'Successfully updated your profile');
    } else {
        $request->session()->flash('error', 'Please try again!');
    }

    return redirect()->back();
}


    // Order
    public function orderIndex()
{
    $orders = Order::orderBy('id', 'DESC')->where('user_id', auth()->user()->id)->paginate(10);
    $products = Product::all();
    return view('user.order.index', compact('orders', 'products'));
}

    public function userOrderDelete($id)
    {
        $order=Order::find($id);
        if($order){
            if($order->status=="process" || $order->status=='delivered' || $order->status=='cancel'){
                return redirect()->back()->with('error','You cannot delete this order now');
            }
            else{
                $status=$order->delete();
                if($status){
                    request()->session()->flash('success','Order successfully deleted');
                }
                else{
                    request()->session()->flash('error','Order could not be deleted');
                }
                return redirect()->route('user.order.index');
            }
        }
        else{
            request()->session()->flash('error','Order not found');
            return redirect()->back();
        }
    }


    public function userOrderReturn(Request $request, $id)
    {
        $order = Order::findOrFail($id);
        $order->return_status = $request->input('return_status', 0); // Default to 0 if not provided
        $order->save();

        return redirect()->back()->with('success', 'Order return status updated successfully.');
    }

    public function userOrderCancel(Request $request, $id)
{
    // Find the order by its ID
    $order = Order::findOrFail($id);

    // Update the cancel_status attribute of the order based on the form input
    $order->cancel_status = $request->input('cancel_status', 0); // Default to 0 if not provided

    // Save the updated order
    $order->save();

    // Redirect back to the previous page with a success message
    return redirect()->back()->with('success', 'Order cancel status updated successfully.');
}



    public function orderShow($id)
    {
        $order=Order::find($id);
        return view('user.order.show')->with('order',$order);
    }

    // Product Review
    public function productReviewIndex(){
        $reviews=ProductReview::getAllUserReview();
        return view('user.review.index')->with('reviews',$reviews);
    }

    public function productReviewEdit($id)
    {
        $review=ProductReview::find($id);
        return view('user.review.edit')->with('review',$review);
    }

    public function productReviewUpdate(Request $request, $id)
    {
        $review=ProductReview::find($id);
        if($review){
            $data=$request->all();
            $status=$review->fill($data)->update();
            if($status){
                request()->session()->flash('success','Review updated');
            }
            else{
                request()->session()->flash('error','Something went wrong! Please try again!');
            }
        }
        else{
            request()->session()->flash('error','Review not found!');
        }

        return redirect()->route('user.productreview.index');
    }

    public function productReviewDelete($id)
    {
        $review=ProductReview::find($id);
        $status=$review->delete();
        if($status){
            request()->session()->flash('success','Review deleted');
        }
        else{
            request()->session()->flash('error','Something went wrong! Try again');
        }
        return redirect()->route('user.productreview.index');
    }

    // Comments
    public function userComment()
    {
        $comments=PostComment::getAllUserComments();
        return view('user.comment.index')->with('comments',$comments);
    }

    public function userCommentDelete($id){
        $comment=PostComment::find($id);
        if($comment){
            $status=$comment->delete();
            if($status){
                request()->session()->flash('success','Comment deleted');
            }
            else{
                request()->session()->flash('error','Error occurred, please try again');
            }
            return back();
        }
        else{
            request()->session()->flash('error','Comment not found');
            return redirect()->back();
        }
    }

    public function userCommentEdit($id)
    {
        $comments=PostComment::find($id);
        if($comments){
            return view('user.comment.edit')->with('comment',$comments);
        }
        else{
            request()->session()->flash('error','Comment not found');
            return redirect()->back();
        }
    }

    public function userCommentUpdate(Request $request, $id)
    {
        $comment=PostComment::find($id);
        if($comment){
            $data=$request->all();
            $status=$comment->fill($data)->update();
            if($status){
                request()->session()->flash('success','Comment updated');
            }
            else{
                request()->session()->flash('error','Something went wrong! Please try again!');
            }
            return redirect()->route('user.post-comment.index');
        }
        else{
            request()->session()->flash('error','Comment not found');
            return redirect()->back();
        }
    }

    // Password Change
    public function changePassword(){
        return view('user.layouts.userPasswordChange');
    }

    public function changPasswordStore(Request $request)
    {
        $request->validate([
            'current_password' => ['required', new MatchOldPassword],
            'new_password' => ['required'],
            'new_confirm_password' => ['same:new_password'],
        ]);

        User::find(auth()->user()->id)->update(['password'=> Hash::make($request->new_password)]);

        return redirect()->route('user')->with('success','Password changed successfully');
    }
}
