<?php

namespace App\Http\Controllers;

use App\Models\SubContent;
use App\Models\Submenu;
use Illuminate\Support\Str;
use Illuminate\Http\Request;

class SubContentController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index($id)
    {
        $submenu = Submenu::findOrFail($id);
        $subContents = SubContent::where('sub_id', $id)->latest()->paginate(10);
        return view('backend.subcontents.index', compact('submenu', 'subContents'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create($id)
    { 
        $submenu = Submenu::findOrFail($id);

        if (!$submenu) {
            // Handle the case where the menu does not exist
            return redirect()->back()->with('error', 'Submenu not found.');
        }
        return view('backend.subcontents.create', compact('id','submenu'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request, $id)
    {
        $request->validate([
            'title' => 'nullable|string',
            'heading' => 'nullable|string',
            'description' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpg,png,jpeg,gif',
        ]);
    
        $imagePath = null;
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $imageName = time() . '.' . $image->getClientOriginalExtension();
            $image->move(public_path('admin/img/'), $imageName);
            $imagePath = 'admin/img/' . $imageName;
        }
    
        SubContent::create([
            'sub_id' => $id,
            'title' => $request->title,
            'slug' => Str::slug($request->title),
            'heading' => $request->heading,
            'image' => $imagePath,
            'description' => $request->description,
            'status' => $request->status ? 1 : 0,
        ]);
    
        return redirect()->route('backend.subcontents.index', ['id' => $id])->with('success', 'Sub Menu Content created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(SubContent $subContent)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id, SubContent $subContent)
{
    return view('backend.subcontents.edit', compact('id', 'subContent'));
}

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id, SubContent $subContent)
{
    $request->validate([
        'title' => 'nullable|string',
        'heading' => 'nullable|string',
        'description' => 'nullable|string',
        'status' => 'nullable|boolean',
        'image' => 'nullable|image|mimes:jpg,png,jpeg,gif',
    ]);

    if ($request->hasFile('image')) {
        if ($subContent->image && file_exists(public_path($subContent->image))) {
            unlink(public_path($subContent->image));
        }

        $image = $request->file('image');
        $imageName = time() . '.' . $image->getClientOriginalExtension();
        $image->move(public_path('admin/img'), $imageName);
        $subContent->image = 'admin/img/' . $imageName;
    }

    $subContent->update([
        'sub_id' => $id,
        'title' => $request->title,
        'slug' => Str::slug($request->title),
        'heading' => $request->heading,
        'url' => $request->url,
        'description' => $request->description,
        'status' => $request->status ? 1 : 0,
        'image' => $subContent->image ?? $subContent->getOriginal('image'),
    ]);

    return redirect()->route('backend.subcontents.index', ['id' => $id])
                     ->with('success', 'Sub Menu Content updated successfully.');
}


    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id, SubContent $subContent)
{
    // Delete image if it exists
    if ($subContent->image && file_exists(public_path($subContent->image))) {
        unlink(public_path($subContent->image));
    }

    // Delete the subContent record
    $subContent->delete();

    // Redirect back with success message
    return redirect()->route('backend.subcontents.index', ['id' => $id])
                     ->with('success', 'Sub Menu Content deleted successfully.');
}


}
