<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\Hash;
use Illuminate\Http\Request;
use App\User;

class UsersController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $users = User::orderBy('id', 'ASC')->paginate(10);
        return view('backend.users.index', compact('users'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('backend.users.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:30',
            'email' => 'required|string|unique:users',
            'password' => 'required|string',
            'address1' => 'required|string|max:255',
            'role' => 'required|in:admin,user',
            'status' => 'required|in:active,inactive',
            'image' => 'required|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'question' => 'required|string',
            'answer' => 'required|string',
        ]);

        $user = new User();
        $user->name = $validated['name'];
        $user->email = $validated['email'];
        $user->address1 = $validated['address1'];
        $user->password = Hash::make($validated['password']);
        $user->role = $validated['role'];
        $user->question = $validated['question'];
        $user->answer = $validated['answer'];
        $user->status = $validated['status'];

        // Handle the image upload
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $filename = time() . '.' . $image->getClientOriginalExtension();
            $image->move(public_path('backend/img'), $filename);
            $user->photo = $filename;
        }

        // Save the user
        $saved = $user->save();

        if ($saved) {
            return redirect()->route('users.index')->with('success', 'User added successfully');
        } else {
            return redirect()->back()->with('error', 'Error occurred while adding user');
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $user = User::findOrFail($id);
        return view('backend.users.edit', compact('user'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $user = User::findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:30',
            'email' => 'required|string',
            'address1' => 'required|string|max:255',
            'role' => 'required|in:admin,user',
            'status' => 'required|in:active,inactive',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'question' => 'required|string',
            'answer' => 'required|string',
        ]);

        $user->name = $validated['name'];
        $user->email = $validated['email'];
        $user->address1 = $validated['address1'];
        $user->role = $validated['role'];
        $user->question = $validated['question'];
        $user->answer = $validated['answer'];
        $user->status = $validated['status'];

        // Handle the image update
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $filename = time() . '.' . $image->getClientOriginalExtension();
            $image->move(public_path('backend/img'), $filename);
            $user->photo = $filename;
        }

        $saved = $user->save();

        if ($saved) {
            request()->session()->flash('success', 'User updated successfully');
        } else {
            request()->session()->flash('error', 'Error occurred while updating user');
        }

        return redirect()->route('users.index');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $user = User::findOrFail($id);
        $deleted = $user->delete();

        if ($deleted) {
            request()->session()->flash('success', 'User deleted successfully');
        } else {
            request()->session()->flash('error', 'Error occurred while deleting user');
        }

        return redirect()->route('users.index');
    }
}
